import time
import argparse
import traceback
import pyvisa as vs
import matplotlib.pyplot as plt
from timeit import default_timer as timer
import pandas as pd
import pyvisa as vs

# Number of bytes per data point
BYTES_PER_DATA_POINT = 4
# USBTMC response byte lengths
USBTMC_MSG = 2048
USBTMC_HEADER = 12
USBTMC_DATA = USBTMC_MSG - USBTMC_HEADER
USBTMC_MAX_RESPONSE_COUNT = 10

# USBTMC device connection variables
SANTEC_VID = 'USB0::0x2428'  #0x2428 is the santec vendor ID (VID)
OPM_200_PID ='D00C'
TIMEOUT_MS = 5000

# sign bit 31 | exponent bits 30-23 | mantissa bits 22-0
def ieee754_to_decimal(ieee):
   sign = -1.0 if (ieee>>31)&1 else 1.0
   #subtract 127 to get the actual exponent
   exponent = (ieee >> 23) & 0xFF
   exponent=exponent-127
   mantissa= ieee & 0x7FFFFF
   # Convert integer to binary string
   # bin() returns the string with '0b' prepended, so start at index 2
   # make sure to add back in the leading 0s
   binary_mantissa = bin(mantissa)[2:].rjust(23, '0')
   decimal_fraction = 0.0
   power = -1
   for bit in binary_mantissa:
        if bit == '1':
            decimal_fraction += 2**power
        power -= 1
   # mantissa leading 1. is omitted in ieee754, add it back in
   return  sign*(2**exponent)*(1.0+decimal_fraction) 

def validateArgs(argument,instrument):
    TRIG_LIST = ["SME", "FREE"]
    MAX_SAMPLES=128000
    AVG_LIST=["us","ms"]

    #check trig mode is valid
    if (not (any(map(argument.mode.__contains__, TRIG_LIST)))):
        print("Mode argument is not recognized. Use SME or FREE.")
        return False

    #check samples are valid
    if int(argument.samples) >MAX_SAMPLES:
        print("Samples must be <128 000.")
        return False
    
    # check avg time is valid
    if (not (any(map(argument.avg.__contains__, AVG_LIST)))):
        print("Avg. time argument is not recognized. Use us or ms.")
        return False

    # Check Power meter number
    if (int(args.detector) < 0):
        print("Invalid number for power meter entered.\n")
        return False

def connectInstrument():
    instrument=None
    rm = vs.ResourceManager()
    # if --ip is given, then use it
    if args.ip != None:
        TCP_INST="TCPIP::"+args.ip+"::5025::SOCKET"
        instrument = rm.open_resource(TCP_INST, open_timeout=5000)
    #otherwise assume USB connection
    else:
        instruments = rm.list_resources()
        # See if any santec USB instruments are present
        if (any(s.startswith(SANTEC_VID) for s in instruments)):
            r = [s for s in instruments if s.startswith(SANTEC_VID) and OPM_200_PID in s ]
            # if at least one opm-200 is present, connect to the first one
            if (len(r) > 0):
                instrument = rm.open_resource(r[0], open_timeout=TIMEOUT_MS)
    
    if instrument !=None:
        # read/write terminations are required for Santec products
        instrument.write_termination = '\n'
        instrument.read_termination = '\n'
        instrument.timeout=TIMEOUT_MS

    return instrument

def setParameters(instrument,arguments):
    # Make sure log mode is cleared before starting to prevent state issues 
    instrument.query("SENSe" + arguments.detector + ":FUNCtion:STATe LOGG,STOP;*OPC?")

    # Set the trigger mode to FREErun (1 trigger to start) or SMEasure (1 trigger per datapoint)
    instrument.query("SENSe" + arguments.detector + ":TRIG:IN " + arguments.mode+";*OPC?")

    # Set dataPoints and averaging time
    instrument.query("SENSe" + arguments.detector + ":FUNCtion:PAR:LOGG "+arguments.samples+","+arguments.avg+";*OPC?")

    # logging should only be done in fixed gain mode (ranges are 8,-10,-30,-50,-70)
    instrument.query("SENSe" + arguments.detector + ":POWer:RANGe "+arguments.range+";*OPC?")

    #double check the settings have been correctly applied
    params = instrument.query(":SENSe"+args.detector+":FUNCtion:PARameter:LOGGing?").split(',')
    samples = params[0]
    avgTime = float(params[1].split(" ")[1])
    print("Number of datapoints:", samples)
    print("Average Time:", avgTime, "ms")

def WaitLoggingComplete(instrument,arguments):
    #poll status while waiting for logging to complete
    while True:
        response = instrument.query("SENSe" + arguments.detector + ":FUNCtion:STATe?").split(',')
        print(response)
        if (int(response[1]) == 0):
            return
        #important to delay a bit so as not to poll too quickly
        time.sleep(1)

def getTCPData(instrument,logSize,arguments):
    dBmList=[]
    lengthRead=0
    #tell the OPM-200 to start sending data back to us
    opm.write(":SENSe"+arguments.detector+":FUNCtion:RESult:BLOCK? 0")
    #for Ethernet connection all of the data is available after in the first block
    while (lengthRead < logSize  - BYTES_PER_DATA_POINT):
        try:
            #read the data in bytes from the instrument and convert to dBm
            dataChunk = instrument.read_bytes(4)
            lengthRead = lengthRead + len(dataChunk)
            dbm = int.from_bytes(dataChunk, byteorder='little')
            ddbm=ieee754_to_decimal(dbm)
            dBmList.append(ddbm)

        except Exception as e:
            print("Error: %s", traceback.format_exc())
            return None
    return dBmList

def getUSBData(instrument,logSize,arguments):
    index = 0
    dBmList=[]
    #append all data to a byte string, this can be easily split after
    dataChunkArray=b''
    remainder = logSize % USBTMC_DATA
    while ((index * USBTMC_DATA) < logSize - remainder):        
        #tell the OPM-200 to start sending data back to us
        opm.write(":SENSe"+arguments.detector+":FUNCtion:RESult:BLOCK? 0")
        # over USB, data is only available in chunks of 10*USBTMC_DATA sized blocks (10*2048 bytes)
        # must read mulitple of them until all data is returned

        # Determine max amount that can be read
        if ((logSize -(index * USBTMC_DATA)) / USBTMC_DATA >= USBTMC_MAX_RESPONSE_COUNT):
            # Read all of the bytes in this block
            for i in range(USBTMC_MAX_RESPONSE_COUNT):
                datachunk = instrument.read_bytes(USBTMC_DATA)
                dataChunkArray+=datachunk

            # Increment index by the number of packets received
            index += USBTMC_MAX_RESPONSE_COUNT
        else:
            while ((logSize -(index * USBTMC_DATA) - remainder) >= USBTMC_DATA):
                datachunk = instrument.read_bytes(USBTMC_DATA)
                dataChunkArray+=datachunk
                index += 1

            if (remainder > 0):
                # Read last "remainder" bytes that are not a full block
                datachunk = instrument.read_bytes(remainder)
                dataChunkArray+=datachunk

    if (index == 0 and remainder > 0):
        # still missing some data, ask for the last bytes
        opm.write(":SENSe"+arguments.detector+":FUNCtion:RESult:BLOCK? 0")
        datachunk = instrument.read_bytes(remainder)
        dataChunkArray+=datachunk

    #got all data, now convert binary string to list 
    L = [dataChunkArray[i:i+4] for i in range(0,len(dataChunkArray),4)]
    #convert list to dBm
    for b in L:
        dbm = int.from_bytes(b, byteorder='little')
        ddbm = ieee754_to_decimal(dbm)
        dBmList.append(ddbm)

    return dBmList

def plotandSaveData(dBmList):
    fig, ax = plt.subplots(figsize=(10, 6))
    x=range(0,len(dBmList))
    df=pd.DataFrame(dBmList)
    df.to_csv("OPM-200-log-data.csv")

    # Create scatter plot
    ax.scatter(x,dBmList,marker='.')

    plt.title("OPM-200 Datalog")
    plt.xlabel("Point")
    plt.ylabel("Power (dBm)")
    plt.show()


#===========================main=========================================================================
# Create argument parser
parser = argparse.ArgumentParser()
parser.add_argument(
    "--detector", type=str, required=True)  # Used to set the power meter number, 0 is the first detector
parser.add_argument(
    "--samples", type=str, required=False,default='1000')  # Used to set desired # of datapoints
parser.add_argument(
    "--avg", type=str, required=False,default='1ms') # set desired averaging time. Recommended <=1ms
parser.add_argument(
    "--range", type=str, required=False,default='10')  # set desired gain range. Valid values are 10,-10,-30,-50,-70
parser.add_argument(
    "--mode", type=str, required=True)  # Used to set trig mode. FREE (single trigger to start) or SME (one trigger per data point)
parser.add_argument(
    "--ip", type=str, required=False)  # Used to connect to the correct OPM-200 if TCP/IP

#all arguments now exist as parameters of the args object
args = parser.parse_args()


#initialize pyvisa to connect to the instrument
opm=connectInstrument()
if opm==None:
    print("OPM-200 not found")
    quit()

#exit script if arguments are not valid
validateArgs(args,opm)

setParameters(opm,args)

# initiate a logging session. After this command the instrument will look for incoming triggers
opm.write("SENSe" + args.detector + ":FUNCtion:STATe LOGGing,STARt")

#uncomment the 2 lines below to send a software trigger instead of waiting for a real one (FREE mode only)
#time.sleep(1)
#opm.write("SENSe" + args.detector + ":TRIGger:SOFTware")
WaitLoggingComplete(opm,args)

#logging complete. 
result = opm.query("SENSe" + args.detector + ":FUNCtion:RESult:LOGSize?")
logSize = int(result)
print("Log size read:", logSize, "\n")

data=[]
if args.ip !=None:
    data=getTCPData(opm,logSize,args)
else:
    data=getUSBData(opm,logSize,args)

# plot the data
plotandSaveData(data)

